package RequestForm::Validate;

# hides DFV internals from caller - returns hashref with keys depending on pass/fail:
# success: 'pass' flag + optional data hashref 
# failure: errs

use Moo;
# apply Role::PDS *1st* and seperately so its methods will be retained when
# applying the LIMS::Controller::Roles::PatientDemographics methods:
with 'RequestForm::Role::PDS'; # method overrides for C::R::PatientDemographics
with 'LIMS::Controller::Roles::PatientDemographics'; # some methods overriden above

use Local::MooX::Types qw(HashReference Object);

has settings => ( is => 'ro', isa => HashReference, required => 1 );
has session  => ( is => 'ro', isa => Object, required => 1 );

use Data::FormValidator;
use Data::FormValidator::Constraints qw(:closures);

use LIMS::Local::Utils;
use Data::Dumper::Concise;

sub validate_form { # returns hashref containing errs (fail) or data (pass) keys
    my $self = shift;
    my $vars = shift;
    
    my $defaults = _dfv_defaults();
    my $dfv = Data::FormValidator->new({}, $defaults);
    
    my $dfv_profile = _dfv_profile(); # DEBUG($dfv_profile);
    my $results = $dfv->check($vars, $dfv_profile); # DEBUG($results);

    my %h;
    if ( $results->has_invalid or $results->has_missing ) { # warn Dumper $results;
        $h{errs} = $results->msgs;
    }
    else {
        my $data = $results->valid; # warn Dumper $data;

        # create dob as DateTime from day, month & year vals & delete original vals:
        my @date_fields = qw(year month day);
        # send $data to_datetime() - method needs a day, month & year hashref entry
        $data->{dob} = LIMS::Local::Utils::to_datetime($data); # warn $data->{dob};
        delete $data->{$_} for @date_fields; # warn Dumper $data;

        # get result of PDS lookup (unless set to skip):
        unless ( $vars->{_skip_pds} ) {
            my $pds_result = $self->get_pds_data($data); # warn Dumper $pds_result;
            $h{pds_result} = $pds_result;
        }

        $h{data} = $data;        
        $h{pass} = 1; # add a 'pass' flag for callers convenience
    }
    return \%h;
}

sub validate_nhs_number { # returns hashref containing errs (fail) or pass keys
    my $self = shift;
    my $vars = shift;
    
    my $defaults = _dfv_defaults();
    my $dfv = Data::FormValidator->new({}, $defaults);
    
    my $dfv_profile = _nhs_number_profile();
    my $results = $dfv->check($vars, $dfv_profile); # DEBUG($results);

    return ( $results->has_invalid or $results->has_missing )
        ? { errs => $results->msgs }
        : { pass => 1 };
}

sub _nhs_number_profile {
    return {
        required => 'nhs_number',
        constraint_methods => {
            nhs_number => _check_nhs_number(),
        }        
    }
}
sub _dfv_profile {
    my @required = qw(
        last_name first_name nhs_number gender location_name location_id
        referrer day month year specimen report_to treatment clinical_details
        taken_by contact datetime doi tb previous hb wbc plt 
    );
    my @optional = qw( middle_name patient_number sample_ref neut lymph other
        pds_code );
    return {
        required => \@required,
        optional => \@optional,
		field_filters => {
			last_name   => 'uc',
			first_name  => 'lc',
			location    => 'lc',
			referrer    => 'lc',
            month       => sub { sprintf '%02d', shift }, # ensure zero-padded
        },
        constraint_methods => {
            nhs_number => _check_nhs_number(),
            year => _check_date(), # or any required date field
        },
		msgs => {
			constraints => { },
        },
    }
}

sub _dfv_defaults {
    return {
		missing_optional_valid => 1,
		filters => 'trim', # trims white space either side of field param
		msgs => {
			any_errors => 'dfv_errors', # default err__
			format 	   => '<div class="dfv-err text-lowercase">%s</div>',
		},
	};
}

sub _check_nhs_number {
	return sub {
		my $dfv = shift; # warn Dumper $dfv;
		my $nhs_number = shift || return 0; # warn 'NHSNo:'.$nhs_number;
		
		# check_nhsno() return true if valid:
		my $is_valid = LIMS::Local::Utils::check_nhsno( $nhs_number ); # warn $is_valid;
		return $is_valid;
    };   
}

sub _check_date {
	return sub {
		my $dfv = shift; # warn Dumper $dfv;

		my $data = $dfv->get_filtered_data; # warn Dumper $data;
        my $date = join '/', @{$data}{qw/day month year/}; # warn $date;
        my $is_valid = LIMS::Local::Utils::check_date($date); # warn $is_valid;
		return $is_valid;
	};    
}

1;