package Routes;

=begin
  # to force Email::Sender transport:
  export MY_EMAIL_SENDER_TRANSPORT=SMTP
  export MY_EMAIL_SENDER_TRANSPORT=Test
  export MY_EMAIL_SENDER_TRANSPORT=MBox

  dev box MTA for smtp is exim4; useful commands:
    sudo mailq
    tail -f /var/log/exim4/mainlog
=cut

use Dancer2;
# use Dancer2::Plugin::Email; # needs all msg construction to be done in route

# override config db_name if supplied as command-line param (set before creating $model):
set db_name => $ENV{dbname} if $ENV{dbname}; # warn setting('db_name');

use Model;
use Data::Printer;
use Time::HiRes qw(gettimeofday tv_interval);

my $model = Model->new( app_config => config() ); # or do:
our $VERSION = $model->version;

# for routes that just return a template of same name:
set auto_page => 1;

# disable sql_trace if test script (needs explicit export SQL_TRACE=1 if req'd):
set enable_sql_trace => 0 if $ENV{HARNESS_VERSION};

use vars qw($t0);

# hooks =========================================================================
hook before => sub {
    # discard old db connection if mysql has gone away since last request:
    $model->db->_check_db_connection; # Local::DB - creates new DBIx::Simple object on demand
    # enable querylogger 1st if configured:
    $ENV{SQL_TRACE} = setting('enable_sql_trace');

    $t0 = [gettimeofday]; # set t0
    # all routes need authenticated session (except /login request):
    unless ( session 'user' ) {
        my $request_path = request->path_info; # p $request_path;
        redirect '/login' unless grep $request_path eq '/'.$_, qw(login index);
    }
};

hook before_template_render => sub {
    my $tokens = shift;
    $tokens->{app_version} = $VERSION;
    $tokens->{runtime} = 1000 * tv_interval $t0, [gettimeofday]; # script runtime msec
    # warn $tokens->{runtime} if setting('environment') eq 'development';
};

#hook after_template_render => sub { warn 1000 * tv_interval $t0, [gettimeofday] };
#hook after => sub { warn 1000 * tv_interval $t0, [gettimeofday] };

# routes =======================================================================
get '/' => sub {
    my $user = session 'user'; # p $user;
    my $data = $model->get_outreach_data($user->{patient_id}); # p $data;
    template welcome => { data => $data  }; # data = href of packs & results
};

get  '/login' => sub { template login => {} }; # so config loaded for webapp_name
post '/login' => sub {
    my $params = params; # p $params;

    # validate login (returns user profile or empty):
    if ( my $user = $model->authenticate_user($params) ) { # p $user;
        session user => $user;
        redirect '/';
    }
    else {
        template login => { error => 'invalid username or password' };
    }
};

post '/ajax_amend_details' => sub { # email web-form to admin:
    my $session = session->read('user'); # p $session;
    my $params  = params; # p $params;

    { # validate web-form password matches session passwd:
        my $result = $model->validate_password($params, $session);
        if ( my $err = $result->{errors}) { # p $err; # href of error(s)
            my $json = to_json({ success => undef, errors => $err });
            return $json;
        }
    }
    { # passwords OK, send email to admin:
        my $result = $model->email_patient_demographics($params, $session);
        if ( my $err = $result->{errors}) { # p $err; # href of error(s)
            return to_json({ success => undef, errors => $err });
        }
        # ok, no error, return good news:
        return to_json({ success => 1, message => $result->{msg} });
    }
};

get '/logout' => sub {
    app->destroy_session;
    redirect('/'); # should redirect to /login
};

get '/gitlog' => sub { template gitlog => { log => $model->gitlog } };

# original index page:
get '/index' => sub { template index => {}, { layout => 'index' } };

true;
