package LogReport;

=begin # from:
http://search.cpan.org/~markov/Log-Report-1.12/lib/Log/Report.pod#The_Reason_for_the_report

program keywords----------------------------------------------------------------
. trace (debug, program)
    The message will be used when some logger has debugging enabled. The messages
	show steps taken by the program, which are of interest to the developers and
	maintainers of the code, but not for end-users.

. assert (program)
    Shows an unexpected condition, but continues to run. When you want the program
	to abort in such situation, use panic.

. info (verbose, program)
    These messages show larger steps in the execution of the program. Experienced
	users of the program usually do not want to see all these intermediate steps.
	Most programs will display info messages (and higher) when some verbose flag
	is given on the command-line.

. notice (program)
    An user may need to be aware of the program's accidental smart behavior, for
	instance, that it initializes a lasting Desktop directory in your home directory.
	Notices should be sparse.

. warning (program)
    The program encountered some problems, but was able to work around it by smart
	behavior. For instance, the program does not understand a line from a log-file,
	but simply skips the line.

. panic (program)
    All other report classes are expected: some predictable situation is encountered,
	and therefore a message is produced. However, programs often do some internal
	checking. Of course, these conditions should never be triggered, but if they
	do... then we can only stop. For instance, in an OO perl module, the base
	class requires all sub-classes to implement a certain method. The base class
	will produce a stub method which triggers a panic when called. The non-dieing
	version of this test is assert.

user keywords-------------------------------------------------------------------
. mistake (user)
    When a user does something wrong, but is correctable by smart behavior of
	the program. For instance, in some configuration file, you can fill-in "yes" or
	"no", but the user wrote "yeah". The program interprets this as "yes", producing
	a mistake message as warning. It is much nicer to tell someone that he/she made
	a mistake, than to call that an error.

. error (user)
    The user did something wrong, which is not automatically correctable or the
	program is not willing to correct it automatically for reasons of code quality.
	For instance, an unknown option flag is given on the command-line. These are
	configuration issues, and have no useful value in $!. The program will be
	stopped, usually before taken off (??).

system keywords-----------------------------------------------------------------
. fault (system)
    The program encountered a situation where it has no work-around. For instance,
	a file cannot be opened to be written. The cause of that problem can be some
	user error (i.e. wrong filename), or external (you accidentally removed a
	directory yesterday). In any case, the $! ($ERRNO) variable is set here.

. alert (system)
    Some external cause disturbs the execution of the program, but the program
	stays alive and will try to continue operation. For instance, the connection
	to the database is lost. After a few attempts, the database can be reached
	and the program continues as if nothing happened. The cause is external, so $!
	is set. Usually, a system administrator needs to be informed about the problem.

. failure (system)
    Some external cause makes it impossible for this program to continue. $! is
	set, and usually the system administrator wants to be informed. The program
	will die. The difference with fault is subtle and not always clear. A fault
	reports an error returned by an operating system call, where the failure would
	report an operational problem, like a failing mount.
=cut

use Log::Report ();

use Data::Dumper;
use Data::Printer colored => 1; # for np() output

use Dancer2;
use Dancer2::Plugin::LogReport;
use Model;

our $VERSION = '0.1';

hook before => sub { # trace 'before hook';
	my $vars = params; # p $vars if setting('environment') eq 'development';
	trace np($vars); # does same as above - uses LogReport 'mode' config setting

};
hook before_template => sub { # trace 'before_template hook';
	# p session;
};

get '/' => sub { template 'index', {}, { layout => 'index' } };

get '/test' => sub { template 'test' };

get '/update' => sub {
	# Log a debug message (console only):
	trace 'started function update()';

    my %values = (
		description => param('description'),
		title       => param('title'),
    );
    # if ( Model->update(\%values) ) { # relies on return value; error redirects to '/'
    if ( process sub { Model->update(\%values) } ) { # doesn't need return value & avoids redirect
	    success 'Success: update OK'; # console and tt output
	}
    else { # only get here if process() used above, otherwise error in update redirects to '/'
        # warning "update failed"; # don't need this - it's set in update()
    }

    notice 'about to render template ....';
  	template 'test';
};

get '/success' => sub { # console and tt output
    # Send a routine notice to the end user
	my $msg = Model->messages('success');
    success $msg;
  	template 'test';
};

get '/notice' => sub { # console and tt output
    # Send a routine notice to the end user
	my $msg = Model->messages('notice');
    notice $msg; # p config;
  	template 'test';
};

get '/mistake' => sub { # console and tt output
	# Warn the user (not fatal)
	my $msg = Model->messages('mistake');
	mistake $msg;
  	template 'test';
};

get '/assert' => sub { # console only, no tt output
	my $msg = Model->messages('assert');
	# assert $msg; # reports wrong line
    report ASSERT => $msg; # reports *this* line,
  	template 'test';
};

get '/info' => sub { # console only, no tt output
	my $msg = Model->messages('info');
    info $msg;
  	template 'test';
};

get '/warning' => sub { # console and tt output
	my $msg = Model->messages('warning');
    warning $msg;
  	template 'test';
};

# 'FAULT', 'ALERT', 'FAILURE', 'PANIC' all fatal errors:
get '/error' => sub { # redirects to '/'
	my $msg = Model->messages('error');
	error $msg; template 'test';
};
get '/alert' => sub { # redirects to '/'
	my $msg = Model->messages('alert');
    alert $msg; template 'test';
};
get '/panic' => sub { # redirects to '/'
	my $msg = Model->messages('panic');
	panic $msg; template 'test';
};
get '/fault' => sub { # redirects to '/'
	my $msg = Model->messages('fault');
    fault $msg; template 'test';
};
get '/failure' => sub { # redirects to '/'
	my $msg = Model->messages('failure');
    failure $msg; template 'test';
};

true;
