#!/usr/bin/env perl

#===============================================================================
# compresses and archives deployment edits log-file today.sql
# rotates development.sql, deployment.sql, debug.log & trace.log
# skips empty files using io($file)->size
# run from cron once per day eg 00:00

# see logs/README.txt for file permission settings
#===============================================================================

use strict;
use warnings;
use feature 'say';

use lib '/home/raj/perl5/lib/perl5';
use IO::All;
use Path::Tiny;
use Data::Printer;
use Logfile::Rotate;
use IO::Compress::Gzip qw(gzip $GzipError);

use FindBin qw($Bin); # warn $Bin; exit;
use lib $Bin . '/../../../lib';
use LIMS::Local::ScriptHelpers;

my $path_to_logs = path($Bin . '/../../../logs')->realpath; # say $path_to_logs;
my $archive = $path_to_logs . '/old';

my $tools = LIMS::Local::ScriptHelpers->new();
# get yesterday's date (allows for BST in DT construct):
my $yesterday = $tools->date_subtract(days => 1); # say $yesterday;
# my $yesterday = $tools->time_now; # say $yesterday;
# time format for gz file:
my $archive_time_format = '%Y_%m%d';

# log-files --------------------------------------------------------------------
my $deployment_edits = $path_to_logs . '/today.sql';
my $development_log  = $path_to_logs . '/development.sql';
my $deployment_log   = $path_to_logs . '/deployment.sql';
my $debug_file       = $path_to_logs . '/debug.log';
my $dbi_trace        = $path_to_logs . '/trace.log';
#-------------------------------------------------------------------------------

{ # deployment edits (INSERT/UPDATE/DELETE), for archiving:
    my $o = io($deployment_edits);
    # compress and archive if it has data:
    if ($o->size) {
        my $input  = $o->name; # warn $input;
        my $output = sprintf '%s/%s.gz',
            $archive, $yesterday->strftime($archive_time_format);

        my $status = gzip $input => $output, Minimal => 1 # avoids full path info
            or die "$0 - gzip failed: $GzipError\n";
        # `gzip -c $src > $output`; # system command alternative
        $o->unlink; # re-created on demand ($o->truncate(0) doesn't work - ?? permissions)
    }
}

# rotate deployment, development & debug logs (returns 0 unless file content):
rotate_file($_) for ($deployment_log, $development_log, $debug_file);

=begin sql.log (replaced by yyyy_mmdd_sql.log)
my $sql_log = "$path_to_logs/sql.log";

if ( io($sql_log)->size ) {
    Logfile::Rotate->new(
        File    => $sql_log,        # base filename
        Dir     => $archive,        # move old files to here
        Count   => 365,             # how many to keep before overwriting
        Gzip    => 'lib',           # use Compress::Zlib for compression (recommended)
        Flock   => 'yes',           # flock if supported
        Persist => 'yes',           # copy current logfile chown settings to any new log files
    )->rotate;
}
=cut

# trace.log --------------------------------------------------------------------
if ( io($dbi_trace)->size ) {
    Logfile::Rotate->new(
        File    => $dbi_trace,    # base filename
        Dir     => $archive,      # move old files to here
        Count   => 365,           # how many to keep before overwriting
        Flock   => 'yes',         # flock if supported
        Persist => 'yes',         # copy current logfile chown settings to any new log files
    )->rotate;
}

sub rotate_file {
    my $file = shift;
    return unless io($file)->size; # say "rotating $file";
    Logfile::Rotate->new(
        File    => $file, # base filename
        Count   => 3,     # how many to keep before overwriting
        Flock   => 'yes', # flock if supported
        Persist => 'no',  # don't need archived file to have original ownership
    )->rotate || warn "couldn't rotate $file - $!"; # possibly doesn't capture stderr
}
