#!/usr/bin/perl

use Test::WWW::Mechanize::CGIApp;

use Data::Dumper;

use strict;
use warnings;

use Test::More tests => 54; # use Test::More 'no_plan';

=begin: # tests:
1) try to access user/group-function - check error due to no user groups
2) add some (valid) user groups
3) submit invalid group_name
4) correct group_name to valid but omit other fields
5) edit all 3 user_group fields
6) try to access user/group-function - check error due to no user functions
7) add some (valid) user_functions
8) submit invalid function_name
9) correct & submit valid function name
10) retrieve & edit last new function entry
11) try to access user/group-function - ok
12) allocate 2 new functions to user group
13) change allocated functions
=cut

BEGIN {
    require 't/test-lib.pl';
}

my $mech = get_mech();

do_login();

my $dbh;

eval {
    $dbh = get_dbh() or die 'no database handle recieved from get_dbh';
};

warn $@ if $@;

# need to drop & re-create following tables:
foreach ( qw/user_groups user_group_function/ ) { # user_functions
    drop_and_recreate($_);
}

$mech->get_ok('/admin/user_group-function');      # print_and_exit();

$mech->content_contains(
    get_messages('admin')->{no_groups},
    'error generated due to no user groups defined',
);

$mech->content_lacks(
    'Set default permissions for user groups',
    'OK: user groups form not loaded',
);                                               # print_and_exit();

# add some user groups:
$mech->get_ok('/admin/user_group');              # print_and_exit();

$mech->content_contains(
    'Setup User Groups',
    'user groups form loaded OK',
);

my %groups = (
    1 => {
        name   => 'mla',
        label  => 'Laboratory assistant',
        detail => 'Lab assistant description goes here',
    },
    2 => {
        name   => 'sec',
        label  => 'Medical secretary',
        detail => 'Secretary description goes here',
    },
    3 => {
        name   => 'bms',
        label  => 'BMS/Clinical scientist',
        detail => 'BMS description goes here',
    },
);

foreach (sort keys %groups) { # print $_, "\n";
    $mech->field('group_name',   $groups{$_}{name});
    $mech->field('group_label',  $groups{$_}{label});
    $mech->field('group_detail', $groups{$_}{detail});
    $mech->submit_form();                            # print_and_exit();
}                                                       # print_and_exit();

foreach (sort keys %groups) { # print $_, "\n";
    $mech->content_contains(
        qq!value="$groups{$_}{name}"!,
        "submitted value $_ group_name detected OK",
    );
    $mech->content_contains(
        qq!value="$groups{$_}{label}"!,
        "submitted value $_ group_label detected OK",
    );
    $mech->content_contains(
        qq!value="$groups{$_}{detail}"!,
        "submitted value $_ group_detail detected OK",
    );
}                                                    # print_and_exit();

# submit invalid group_name:
$mech->field('group_name',   'invalid name');
$mech->submit_form();                         # print_and_exit();

has_dfv_errors();

$mech->content_contains(
    'invalid format - no spaces allowed',
    'OK: invalid function_name detected',
);

has_missing();

# correct invalid name:
$mech->field('group_name', 'valid_name');
$mech->submit_form();                         # print_and_exit();

# still have missing data:
has_missing();

# submit valid group label:
$mech->field('group_label',   'group label');
$mech->submit_form();                         # print_and_exit();

# still have missing data:
has_missing();

# submit valid description :
$mech->field('group_detail', 'group detail');
$mech->submit_form();                         # print_and_exit();

# no missing data fields:
lacks_dfv_errors();

$mech->get_ok('/admin/user_group/edit/4');       # print_and_exit();

# correct all 3 fields:
$mech->field('group_name',   'admin');
$mech->field('group_label',   'Administrator');
$mech->field('group_detail', 'Description');
$mech->submit_form();                           # print_and_exit();

foreach ( qw/admin Administrator Description/ ) { # print $_, "\n";
    $mech->content_contains(
        qq!value="$_"!,
        'edited value detected OK',
    );
}

# create new user group
{    
    $mech->submit_form(
        fields => {
            group_name    => 'New',
            group_label   => 'New Label',
            group_detail  => 'admin@new.net',
            active        => 'yes',
        },
    );                                               # print_and_exit();
        
}

{ # edit new group - change all fields:
    foreach ( 'admin@new.net', 'New Label' ) { 
      $mech->content_contains(
          qq!value="$_"!,
          'new user group value detected ok',
      );
    }
    
    $mech->get_ok('/admin/user_group/edit/5');      # print_and_exit();
    
    $mech->content_contains(
        'Admin &raquo; Edit User Group',
        'requested page loaded ok',
    );
    
    foreach ( 'admin@new.net', 'New Label' ) { 
      $mech->content_contains(
          qq!value="$_"!,
          'new user group value detected ok',
      );
    }

    $mech->submit_form(
        fields => {
            group_name    => 'Changed',
            group_label   => 'Changed Label',
            group_detail  => 'admin@changed.net',
            active        => 'yes',
        }
    );                                  # print_and_exit();
    
    # new values detected:
    foreach ('Changed Label', 'admin@changed.net') {
      $mech->content_contains(
          qq!value="$_"!,
          'new user group value detected ok',
      );
    }
    # old values not detected:
    foreach ('New Label', 'admin@new.net') {
      $mech->content_lacks(
          qq!value="$_"!,
          'OK: old user group value not detected',
      );
    }    
}

$mech->get_ok('/admin/user_group-function');     # print_and_exit();

### select 1st menu item :
$mech->select('id', { n => 2 });
$mech->submit_form();                        # print_and_exit();

my %functions = (
    1 => {
        name   => 'screen',
        detail => 'screen requests',
    },
    2 => {
        name   => 'report',
        detail => 'report requests',
    },
    3 => {
        name   => 'search',
        detail => 'do search',
    },
);

=begin # not user-configurable anymore:
$mech->content_contains(
    get_messages('admin')->{no_functions},
    'error generated due to no user functions defined',
);

$mech->content_lacks(
    'Function description',
    'OK: user functions form not loaded',
);

# add some user_functions (disabled):
$mech->get_ok('/admin/user_function');     # print_and_exit();

foreach (sort keys %functions) { # print $_, "\n";
    $mech->field('function_name',   $functions{$_}{name});
    $mech->field('function_detail', $functions{$_}{detail});
    $mech->submit_form();                            # print_and_exit();
}                                                       # print_and_exit();

foreach (sort keys %functions) { # print $_, "\n";
    $mech->content_contains(
        qq!value="$functions{$_}{name}"!,
        "submitted value $_ function_name detected OK",
    );
    $mech->content_contains(
        qq!value="$functions{$_}{detail}"!,
        "submitted value $_ function_detail detected OK",
    );
}

# submit invalid function_name:
$mech->field('function_name',   'invalid name');
$mech->submit_form();                         # print_and_exit();

has_dfv_errors();
$mech->content_contains(
    dfv_format('single_word'),
    'OK: invalid format detected',
);                                                      # print_and_exit();

has_missing();

# submit valid fields:
$mech->field('function_name', 'function_name');
$mech->field('function_detail', 'function detail');
$mech->submit_form();                         # print_and_exit();

# no missing data fields:
lacks_dfv_errors();
=cut

$mech->get_ok('/admin/user_function/edit/1');       # print_and_exit();

# correct both fields:
# $mech->field('function_name',   'search'); # not editable anymore
$mech->field('function_detail', 'do search');
$mech->submit_form();                          # print_and_exit();

foreach ('do search' ) { # print $_, "\n"; # 'errors', 
    $mech->content_contains(
        qq!value="$_"!,
        'edited value detected OK',
    );
}                                               #  print_and_exit();

### Allocate Initial User Group Functions:
$mech->get_ok('/admin/user_group-function');     # print_and_exit();

### select 1st menu item :
$mech->select('id', { n => 2 });
$mech->submit_form();                         # print_and_exit();

foreach (sort keys %functions) { # print $_, "\n";
    $mech->content_contains(
        "<td>$functions{$_}{detail}</td>",
        "value $_ function_detail loaded OK",
    );
    $mech->content_contains(
        qq!input type="checkbox" name="function_id" value="$_"!,
        "value $_ checkbox loaded OK",
    );
}

### select 2nd form:
$mech->form_name('group_function');

# check 1st 2 boxes:
for (1,2) {
    $mech->tick('function_id', $_, 1);
}

$mech->submit_form();                 # print_and_exit();


for (1,2) {
    $mech->content_contains(
        qq!name="function_id" value="$_" checked!,
        "value $_ checkbox checked OK",
    );
}

$mech->content_lacks(
    qq!name="function_id" value="3" checked!,
    "OK: value 3 checkbox not selected",
);

### select 2nd form:
$mech->form_name('group_function');

# remove 2nd option, select 3rd:
$mech->tick('function_id', 2, 0);
$mech->tick('function_id', 3, 1);

$mech->submit_form();              # print_and_exit();

$mech->content_lacks(
    qq!name="function_id" value="2" checked!,
    "value 2 checkbox deselected OK",
);

for (1,3) {
    $mech->content_contains(
        qq!name="function_id" value="$_" checked!,
        "value $_ checkbox checked OK",
    );
}

# do_logout(); logout link not available for admin function
