package Reporter::Model;

use Reporter::Validation; # HTML::FormHandler class
use Reporter::Class; # provides Moo, Modern::Perl, Local::MooX::Types & Data::Printer::p
use Reporter::DB;

has dbname => ( is => 'ro', isa => String, required => 1 ); # hilis4, test, etc

has db => (
    is => 'lazy',
    builder => sub { Reporter::DB->new( dbname => shift->dbname ) },
);

has validator => (
    is => 'lazy',
    builder => sub { Reporter::Validation->new },
);

use Digest::SHA1;
use DateTime::Format::MySQL;
use Lingua::EN::Numbers qw(num2en_ordinal);

sub authenticate_user {
    my ($self, $params) = @_; # p $params;

    my $userid = $params->{userid}; # username or email addr
    my $passwd = $params->{password};

    my $digest = _sha1_digest($passwd); # p $digest;

    if ( my $user = $self->db->get_user_data($userid) ) { # p $user; p $digest;
        return $user if $user->{password} eq $digest;
    }
    # either no user retrieved by username or passwd, or passwd incorrect:
    return 0;
}

sub validate {
    my ($self, $params) = @_; # p $params;

    my $validator = $self->validator;
    my $result = $validator->run( params => $params ); # p $result;

    my %h = (
        errors  => $result->errors, # empty if validation passed
        failed  => $result->has_errors, # true or false
        success => ! $result->has_errors, # ie $ref->{success} or $ref->{failed}
    ); # p %h;
    return \%h;
}

sub load_request {
    my ($self, $lab_number) = @_; # p $lab_number;

    # split lab_number into request_number & yr:
    my ($request_number, $yr) = $lab_number =~ m!(\d+)/(\d{2})!;

    # request data:
    my $request = $self->db->get_request_data($request_number, $yr); # p $request;

    { # section result summaries:
        my $data = $self->db->get_result_summaries($request->{id}); # p $data;
        $request->{result_summaries} = $data;
    }
    { # lab test results:
        my $data = $self->db->get_lab_test_results($request->{id}); # p $data;
        $request->{test_results} = $data;
    }
    # dates to datetime:
    $request->{$_} = $self->to_datetime($request->{$_}) for qw(registered dob);
    return $request;
}

sub get_diagnoses {
    my ($self, $str) = @_; # p $str;

    my $data = $self->db->get_diagnoses($str); # p $data;
    return $data;
}

sub num2ordinal { num2en_ordinal($_[1]) }

sub to_datetime {
    my $self = shift;
    my $date = shift || return;

    return $date =~ /\d{2}:\d{2}:\d{2}$/ # time
        ? DateTime::Format::MySQL->parse_datetime($date)
        : DateTime::Format::MySQL->parse_date($date);
}

sub _sha1_digest {
	my $str = shift; # warn $str;

    my $sha1 = Digest::SHA1->new;
    $sha1->add($str);

    return $sha1->b64digest;
}

1;
