package Reporter::Validator;

# called as $app->model->validator->validate( <profile_name> => $params );

use Data::FormValidator;
use Data::FormValidator::Constraints qw(:closures);

use Reporter::Class;

has dfv => (
    is => 'lazy',
    builder => sub { Data::FormValidator->new( {}, _dfv_defaults() ) },
);

has profiles => (
    is => 'ro',
    isa => HashReference,
    builder => 1,
);

#-------------------------------------------------------------------------------
sub _build_profiles { # hashref of dfv profiles
  {
    search => {
        required => 'lab_number',
        constraint_methods => {
            lab_number => constraint_method_for('lab_num'),
        },
        msgs => {
            constraints => {
                lab_num => 'lab number format incorrect: require nnn/yy',
            },
        },
    },
    report => {
        required => [ qw( status specimen_quality diagnosis_id ) ],
        optional => [ qw(
            request_notes clinical_details secondary_diagnosis_id morphology
            comment )
        ],
		dependencies => {
            _require_biopsy_details => 'biopsy_site',
            _require_gross_description => 'gross_description',
        },
        constraint_methods => { },
        msgs => {
            constraints => { },
        },
    },
  }
}

#-------------------------------------------------------------------------------
sub validate {
    my ($self, $profile_name, $params) = @_; # p $profile_name; p $params;

    my $profile = $self->profiles->{$profile_name}; # ddp $profile;
    my $result  = $self->dfv->check($params, $profile); # ddp $result;

    my %h = (
        dfv_errs => $result->msgs, # empty hashref if success
        valid    => $result->valid     || {}, # undef if failed
        success  => $result->success   || 0,  # empty if failed
        failed   => ! $result->success || 0,  # empty if failed
    ); # ddp %h;
    return \%h;
}

#-------------------------------------------------------------------------------
sub constraint_method_for {
    my $profile_name = shift;
    my $methods = {
        lab_num => sub {
            my $dfv = shift; # ddp $dfv;
            $dfv->name_this('lab_num'); # to allow custom msg
            my $data = $dfv->get_filtered_data; # ddp $data;
            $data->{lab_number} =~ qr{\d+/\d{2}};
        },
    };
    return $methods->{$profile_name};
}

#-------------------------------------------------------------------------------
sub _dfv_defaults {
    return {
		missing_optional_valid => 1,
		filters => 'trim', # trims white space either side of field param
		msgs => {
			# any_errors => 'dfv_errors', # not needed
			format => '%s', # style goes in .tt not here !!
		},
	};
}

1;
