use strict;
use warnings;

BEGIN { # set test env otherwise get development config settings - unless explicitly
    # set at command-line: "export DANCER_ENVIRONMENT=development; prove -lv t/"
    $ENV{DANCER_ENVIRONMENT} ||= $ENV{PLACK_ENV} ||= 'test';
}

use Test::More;
use Test::Most; # bail_on_fail

require_ok( 'Reporter::Test' ); # also loads Reporter::Class - imports Data::Printer, Moo, etc
my $t = Reporter::Test->new;

my $model = $t->app->model;
my $dbix  = $model->db->dbix; # ddp $dbix->dbh->{Name};
my $now   = DateTime->now;

# check test scripts using sqlite in-memory db:
bail_on_fail;
like( $dbix->dbh->{Name}, qr/:memory:/, 'sqlite db in use' );

use lib 't';
use schema;
run_schema($dbix); # create tables & add some data (users, requests, request_specimen, etc)

my $user_id = 1;
{ # userid = username, incorrect password:
    my %params = ( userid => 'queenie', password => 'queenie' );
    my $res = $model->validate_user_credentials(\%params); # ddp $res;
    is($res, 0, 'incorrect password');
}
{ # userid = username, correct password:
    my %params = ( userid => 'queenie', password => 'Queenie' );
    my $res = $model->validate_user_credentials(\%params); # ddp $res;
    is( ref $res, 'HASH', 'validate returned hashref' );
    is( $res->{username}, 'queenie', 'username matches for userid = username' );
    is( $res->{id}, $user_id, 'user id matches' );
}
{ # userid = email, correct password:
    my %params = ( userid => 'queenie@mail.com', password => 'Queenie' );
    my $res = $model->validate_user_credentials(\%params); # ddp $res;
    is( ref $res, 'HASH', 'validate returned hashref' );
    is( $res->{username}, 'queenie', 'username matches for userid = email' );
}
{ # get reports for user #1:
    my $n = get_reports_for_user($user_id);
    is( $n, 0, "no reports for user id $user_id" );
}
{ # add some draft_reports:
    my %h = (
        user_id      => $user_id,
        request_id   => 1, # will auto-increment in loop
        diagnosis_id => 1,
    );
    $dbix->insert('request_draft_report', \%h) && $h{request_id}++ for (1..5);
        # my $ref = $dbix->select('request_draft_report', '*')->hashes; p $ref;
        # my $n = $dbix->count('request_draft_report'); p $n;
}
{ # get reports for user #1:
    my $n = get_reports_for_user($user_id);
    is( $n, 5, "5 reports for user id $user_id" );
}
{ # user reports:
    my $res = $model->user_reports_list($user_id); # p $res;
    is ( ref $res, 'HASH', 'user reports returned hashref' );
    is ( ref $res->{reports}, 'ARRAY', 'reports is in list format' );
    is ( @{ $res->{reports} }, 5, '5 reports returned' );
    is( ref $res->{specimen}, 'ARRAY', 'specimen is in list format' );
    # contains 5 PB's:
    is ( $res->{specimen}->[0]->{id}, 'PB', 'expected sample type' );
    is ( $res->{specimen}->[0]->{n}, 5, 'expected sample number' );
}
{ # new report:
    my %h = (
        biopsy_site              => 'biopsy site',
        clinical_details         => 'clinical details',
        comment                  => 'comment',
        created_at               => $now,
        diagnosis_id             => 1,
        gross_description        => 'gross description',
        morphology               => 'morphology',
        request_id               => 6,
        request_notes            => 'request notes',
        secondary_diagnosis_id   => undef,
        specimen_quality         => 'specimen quality',
        status                   => 'new',
        user_id                  => $user_id,
    );
    my $res = $model->save_report(\%h); # p $res;
    is ( ref $res, 'HASH', 'save report returned hashref' );
    is ( $res->{success}, 1, 'new record created successfully');
    my $ref = $dbix->select('request_draft_report','*', { request_id => 6 })->hash; # p $ref;
    is_deeply( $ref, \%h, 'data structures identical' );
}

done_testing(19);

sub get_reports_for_user {
    my $user_id = shift;
    my %h = (
        user_id => $user_id,
        page    => 1,
        uri     => '/',
    );
    my $res = $model->get_reports_for_user(\%h); # ddp $res;
    my $n = @{ $res->{reports} };
    return $n;
}
