# https://metacpan.org/dist/Dancer2/view/lib/Dancer2/Manual/Testing.pod

use strict;
use warnings;

BEGIN { # set test env otherwise get development config settings - unless explicitly
    # set at command-line: "DANCER_ENVIRONMENT=development prove -lv t/"
    $ENV{DANCER_ENVIRONMENT} ||= $ENV{PLACK_ENV} ||= 'test';
}

use Test::More;
use Plack::Test;
use Data::Printer;
use HTTP::Cookies;
use HTTP::Request::Common;

use_ok('Routes');

my $test = Plack::Test->create( Routes->to_app );
my $jar  = HTTP::Cookies->new();
my $url  = 'http://localhost';

subtest 'Landing page' => sub {
    my $res = $test->request( GET '/' );
    ok( $res->is_redirect, '[GET /] redirect' );
};

# see DraftReporter/t/routes.t =================================================
subtest 'Login' => sub {
    my $request = POST '/login', [ password => 't3st' ] ;
    my $response = $test->request( $request );
    ok( $response->is_redirect, '[POST /login] redirect' );

    # handle cookies for next request:
    $jar->extract_cookies($response);
      # my @cookies;
      # $cookie_jar->scan( sub { @cookies = @_ }); p @cookies;
};

{
    # create HTTP::Request object and add cookies to it:
    my $req = GET $url; # full url to match cookie expectation
    $jar->add_cookie_header( $req ); # p $req->dump;
    # resubmit GET '/', should get 'new_document' link:
    my $res = $test->request( $req ); # p $res;
    like( $res->as_string, qr/new_document/, 'have link for new document' );
}

{
    my %data = ( title => 'title', keywords => 'kwd1, kwd2',
        content => 'some content' );
    my $req = POST $url . '/new_document', \%data; # full url to match cookie expectation
    # don't need to add cookies here as /new_document isn't protected by login
    my $res = $test->request( $req ); # p $res;
    # successful input gets redirect
    ok( $res->is_redirect, '[POST /new_document] redirect' );
}
{
    my $req = GET $url . '/id/1'; # full url to match cookie expectation
    my $res = $test->request( $req ); # p $res;
    like( $res->as_string, qr/kwd1,kwd2/, 'have expected keywords' );
    like( $res->as_string, qr/some content/, 'have expected content' );
}

{
    my %data = (
       title    => 'Ripples They Cause in the World',
       keywords => 'mst, rip',
       content  => q!It is with great sadness that we announce the passing of
            Shadowcat co-founder, Matt S. Trout! );
    my $req = POST $url . '/new_document', \%data; # full url to match cookie expectation
    # don't need to add cookies here as /new_document isn't protected by login
    my $res = $test->request( $req ); # p $res;
    # successful input gets redirect
    ok( $res->is_redirect, '[POST /new_document] redirect' );
}
{
    my $req = GET $url . '/id/2'; # full url to match cookie expectation
    my $res = $test->request( $req ); # p $res;
    like( $res->as_string, qr/mst/, 'have expected keyword' );
    like( $res->as_string, qr/Ripples They Cause in the World/,
         'search by id returned expected content' );
}
# search:
{
    my $req = POST $url . '/search' , [ search => 'mst' ]; # full url to match cookie expectation
    my $res = $test->request( $req ); # p $res;
    like( $res->as_string, qr/Ripples/,
        'search by free-text returned expected content' );
}
# edit:
{
    my $req = POST $url . '/edit/1'; # full url to match cookie expectation
    my $res = $test->request( $req ); # p $res;
    like( $res->as_string, qr{/update/1}, 'contains update link' );
    like( $res->as_string, qr/some content/, 'has expected content' );
}
# submit update:
{
    my $req = POST $url . '/update/1',
        [ content => 'other content', keywords => 'kwd1,kwd3', title => 'title' ];
    my $res = $test->request( $req ); # p $res;
    # successful input gets redirect
    ok( $res->is_redirect, '[POST /new_document] redirect' );
    # have to retrieve update manually due to redirect:
    $req = GET $url . '/id/1'; # full url to match cookie expectation
    $res = $test->request( $req ); # p $res;
    like( $res->as_string, qr/kwd3/, 'has updated keyword' );
    like( $res->as_string, qr/other content/, 'has updated content' );
    # original text not found:
    unlike( $res->as_string, qr/kwd2/, 'lacks original keyword' );
    unlike( $res->as_string, qr/some content/, 'lacks original content' );
}

done_testing (18);