# https://metacpan.org/dist/Dancer2/view/lib/Dancer2/Manual/Testing.pod
# chatgpt HTTP-Cookies.txt
# run: carmel exec prove -lv t/infolib.t

use strict;
use warnings;

BEGIN { # set test env otherwise get development config settings - unless explicitly
    # set at command-line: "DANCER_ENVIRONMENT=development prove -lv t/"
    $ENV{DANCER_ENVIRONMENT} ||= $ENV{PLACK_ENV} ||= 'test';
}

use App::Test; # get_next_location, process_request & initialise
use Test::More;
use Plack::Test;
use Data::Printer;
use HTTP::Cookies;
use HTTP::Request::Common;

use_ok('DocsLib');

my $test = Plack::Test->create( DocsLib->to_app ); # p $test->app; exit;
my $jar  = HTTP::Cookies->new();
my $url  = 'http://localhost/infolib';

App::Test::initialise( $jar, $test );

my %hx_request_headers = (
       HX_Target  => 'content-div',   # ← optional: element id to swap
       HX_Trigger => 'save-button',   # ← optional: id/name of the triggering element          
       HX_Request => 'true',          # ← required by htmx
       HX_Current_URL => '/',         # necessary for correct url in record.tt using hx-current-url header
);

# ==============================================================================
subtest 'Landing page' => sub {
    my $res = $test->request( GET '/infolib/' );
    ok( $res->is_redirect, '[GET /] redirect' );
};

# see DraftReporter/t/routes.t and dancyland blog.t ============================
subtest 'Login' => sub {
    my $request = POST '/login', [ password => 't3st' ] ;
    my $response = $test->request( $request );
    ok( $response->is_redirect, '[POST /login] redirect' );

    # handle cookies for next request:
    $jar->extract_cookies($response);
      # my @cookies;
      # $cookie_jar->scan( sub { @cookies = @_ }); p @cookies;
};

subtest 'Home page' => sub {
    # create HTTP::Request object and add cookies to it:
    my $req = GET $url . '/', # full url to match cookie expectation
        %hx_request_headers;
    $jar->add_cookie_header( $req ); # p $req->dump;
    # resubmit GET '/', should get 'new_document' link:
    my $res = $test->request( $req ); # p $res;
    like( $res->as_string, qr{/infolib/new_document}, 'have link for new document' );
};

subtest 'Create document - 1' => sub {
    my %data = ( title => 'title', content => 'some content' );
    my $req = POST $url . '/create', \%data, # full url to match cookie expectation
        %hx_request_headers;
    my $res = process_request($req);
    # successful input gets redirect
    ok( $res->is_redirect, '[POST /create] redirect' );
    
    # have to retrieve update manually due to redirect:
    my $next = get_next_location($res);
    like( $next->as_string, qr/Input success/, 'has input success' );
    like( $next->as_string, qr/some content/, 'have expected content' );
};

subtest 'Create document - 2' => sub {
    my %data = (
       title    => 'Ripples They Cause in the World',
       content  => q!It is with great sadness that we announce the passing of
            Shadowcat co-founder, Matt S. Trout! );
    my $req = POST $url . '/create', \%data, # full url to match cookie expectation
        %hx_request_headers;
    my $res = process_request($req);
    # successful input gets redirect
    ok( $res->is_redirect, '[POST /create] redirect' );

    # have to retrieve update manually due to redirect:
    my $next = get_next_location($res);
    like( $next->as_string, qr/Input success/, 'has input success' );
    like( $next->as_string, qr/Ripples They Cause in the World/,
         'search by id returned expected content' );
};

subtest 'Search' => sub {
    my $req = POST $url . '/search' , [ search => 'Shadowcat' ], # full url to match cookie expectation
        %hx_request_headers;
	# don't need to add cookies here as route isn't protected by login
    my $res = $test->request( $req ); # p $res;
    like( $res->as_string, qr/Ripples/,
        'search by free-text returned expected content' );
};

subtest 'Edit record' => sub {
    my $req = GET $url . '/edit/1', # full url to match cookie expectation
        %hx_request_headers; # p $req;
	$jar->add_cookie_header( $req );
    my $res = $test->request( $req ); # p $res;
    like( $res->as_string, qr{/update/1}, 'contains update link' );
    like( $res->as_string, qr/some content/, 'has expected content' );
};

subtest 'Update document' => sub {
    my $req = POST $url . '/update/1',
        [ content => 'other content', title => 'title' ],
        %hx_request_headers;
    my $res = process_request($req);
    # successful input gets redirect
    ok( $res->is_redirect, '[POST /update] redirect' );
    
    # have to retrieve update manually due to redirect:
    my $next = get_next_location($res);
    like( $next->as_string, qr/Update success/, 'has update success' );
    like( $next->as_string, qr/other content/, 'has updated content' );
    # original text not found:
    unlike( $next->as_string, qr/some content/, 'lacks original content' );
}; 

subtest 'Test error' => sub {
    my %data = (title => 'title', content => 'content', test_err => 'test error');
    my $req = POST $url . '/create', \%data, %hx_request_headers;
    $jar->add_cookie_header( $req );
    my $res = $test->request( $req ); # p $res;
    like( $res->as_string, qr/test error/, 'has error string' );    
};

done_testing (10);